/*
 * Load an ftc file and dump some info out.
 */

#if defined(_MSC_VER)
#include "err.h"
#else
#include <err.h>
#endif

#include <FreeImage.h>
#include <ATI_Compress.h>

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <io.h>

#include "hexdump.h"
#include "bin_ops.h"
#include "util.h"
#include "ftex.h"
#include "getopt.h"

int quiet = 0;
char *progname;

void usage(void)
{
    printf("dftex: [options] <infile> <outfile>\n"
            "Options:\n"
            "-q                     quiet mode - only errors are printed\n"
            "-m                     also dump mipmaps as seperate files\n"
            "\n"
            "the output filetype will be determined by the extention -  \n"
            "many file types are supported, png is good.\n"
        );
}

uint8_t *decompress_dxt1(int width, int height, uint8_t *buf, unsigned int mipmap_size)
{
  ATI_TC_Texture src, dst;
  ATI_TC_ERROR ati_err;

  src.dwSize = sizeof(src);
  src.dwWidth = width;
  src.dwHeight = height;
  src.dwPitch = 0;
  src.format = ATI_TC_FORMAT_DXT1;
  src.pData = buf;
  src.dwDataSize = ATI_TC_CalculateBufferSize(&src);

  if (src.dwDataSize != mipmap_size)
    errx(EXIT_FAILURE, "%s: data size mismatch\n",
           progname);

  dst.dwSize = sizeof(dst);
  dst.dwWidth = width;
  dst.dwHeight = height;
  dst.dwPitch = width * 4;
  dst.format = ATI_TC_FORMAT_ARGB_8888;
  dst.dwDataSize = ATI_TC_CalculateBufferSize(&dst);
  dst.pData = (ATI_TC_BYTE*) malloc(dst.dwDataSize);

  ati_err = ATI_TC_ConvertTexture(&src, &dst, NULL, NULL, NULL, NULL);

  if (ati_err != ATI_TC_OK)
  {
    free(dst.pData);
    errx(EXIT_FAILURE, "me: error while trying to decompress texture: %d",
           ati_err);
  }

  return(dst.pData);
}

int main (int argc, char **argv)
{
	char *infile, *outfile;
	int fd, tmp;
	uint8_t *buffer, *b_tmp;
	off_t size;
	struct stat finfo;
	char ch;
	uint32_t version, width, height, i, j, mipmap_count, format_count,
           format, where, mipmap_size, do_mips;
  FREE_IMAGE_FORMAT oft = FIF_UNKNOWN;

  do_mips = 0;

  progname = argv[0];

  while ((ch = (char)getopt(argc, argv, "h?mq")) != -1)
  {
      switch (ch) {
        case 'm':
          do_mips = 1;
          break;
        case 'q':
          quiet = 1;
          break;
        case '?':
        case 'h':
        default:
          usage();
          return EXIT_FAILURE;
      }
  }

  argc -= optind;
  argv += optind;

  if (argc != 2)
  {
    usage();
    return EXIT_FAILURE;
  }

	infile = argv[0];
	outfile = argv[1];

  check_fi_version();

  if (!quiet)
    printf("Built using Freeimage version: %s\n", FreeImage_GetVersion());

  FreeImage_SetOutputMessage(FreeImageErrorHandler);

  oft = FreeImage_GetFIFFromFilename(outfile);

  if (oft == FIF_UNKNOWN)
		errx(EXIT_FAILURE, "%s: unknown filetype for outfile %s", progname, outfile);

  if (!FreeImage_FIFSupportsWriting(oft))
		errx(EXIT_FAILURE, "%s: sorry, FreeImage dosn't know how to save %s files",
      progname, FreeImage_GetFormatFromFIF(oft));

	fd = open(infile, O_RDONLY|O_BINARY, 0);

	if (fd < 0)
		errx(EXIT_FAILURE, "%s: unable to open %s", progname, infile);

	fstat(fd, &finfo);

	buffer = malloc(finfo.st_size);

	size = read(fd, buffer, finfo.st_size);

	if (memcmp(buffer, "FTEX", 4) != 0)
	{
		errx(EXIT_FAILURE, "%s: File %s is not an FTEX file\n", progname, infile);
	}

	version = read_32(buffer + 4);
	width = read_32(buffer + 8);
	height = read_32(buffer + 12);
	mipmap_count = read_32(buffer + 16);
	format_count = read_32(buffer + 20);

  if (!quiet)
  {
    printf("%s: FTEX file version %d\n",
      infile, version);

    printf("%s: %dx%d, %d mipmaps, %d formats, %d bytes\n", infile,
      width, height, mipmap_count, format_count, size);
  }
  if (do_mips)
  {
    char * foo;
    foo = strrchr(outfile, '.');
    foo[0] = '\0';
  }

	b_tmp = buffer + 24;
	for (i = 0 ; i < format_count ; i++)
	{
		format = read_32(b_tmp);
		where = read_32(b_tmp + 4);
		if (format == FTEX_FORMAT_DXT1)
		{
		  if (!quiet)
        printf("%s: DXT1 format\n", infile);
		}
		else if (format == FTEX_FORMAT_RGB)
		{
      if (!quiet)
        printf("%s: 24bit RGB format\n", infile);
		}
		else
			errx(EXIT_FAILURE, "unknown format: %d for %s\n", format, infile);

		tmp = where;
		for (j = 1; j < (mipmap_count + 1); j++)
		{
      char *filename;
      uint8_t *dst = NULL;
      int mw, mh, divisor;
      FIBITMAP *outbm = NULL, *tmpbm = NULL;

      if (!do_mips && j > 1)
        break;

			divisor = 2 << (j - 2);

			if (divisor == 0)
				divisor = 1;

      mw = width / divisor;
      mw = mw < 1 ? 1 : mw;

      mh = height / divisor;
      mh = mh < 1 ? 1 : mh;

			mipmap_size = read_32(buffer + tmp);
			if (!quiet)
        printf("mipmapsize: %d level: %d %dx%d\n", mipmap_size, j, mw, mh);

      if (do_mips)
      {
        filename = malloc(strlen(outfile) + 42);
        sprintf(filename, "%s_%02d_%dx%d.%s%c", outfile, j, mw, mh, outfile + strlen(outfile) + 1, 0x00);
      } else {
        filename = outfile;
      }

      if (format == FTEX_FORMAT_DXT1) {
        dst = decompress_dxt1(mw, mh, buffer + tmp + 4, mipmap_size);
        outbm = FreeImage_ConvertFromRawBits(dst, mw, mh, mw * 4, 32,
          FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, 0);
        tmpbm = FreeImage_ConvertTo24Bits(outbm);
        FreeImage_Unload(outbm);
        outbm = tmpbm;

      } else if (format == FTEX_FORMAT_RGB) {
        int j2;
        uint8_t t;

        dst = buffer + tmp + 4;

        for (j2 = 0 ; j2 < mw * mh * 3 ; j2 += 3)
        {
            t = dst[j2];
            dst[j2] = dst[j2 + 2];
            dst[j2 + 2] = t;
        }

        outbm = FreeImage_ConvertFromRawBits(dst, mw, mh, mw * 3, 24,
          FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, 0);
      }

      if (outbm == NULL)
      {
        errx(EXIT_FAILURE, "Failed to convert to a FIB\n");
      }

      if (FreeImage_IsTransparent(outbm))
      {
        printf("WARNING: %s : mipmap %d has transparent bits\n", infile, j);
      }

      if (!FreeImage_Save(oft, outbm, filename, 0))
      {
        printf("WARNING: %s might not of saved properly\n", filename);
      }
      FreeImage_Unload(outbm);

      if (do_mips)
        free(filename);

      if (format == FTEX_FORMAT_DXT1) {
        free(dst);
      }// else if (format == FTEX_FORMAT_RGB) {
//        dst = decompress_rgb(mw, mh, buffer + tmp + 4);
//      }

			tmp += mipmap_size + 4;
		}
	}

	close(fd);

	free(buffer);

	return 0;
}

